 
// ************************************************************************
// Replay FX file for world shading
// Pass: global environment lighting (ambient + directional lighting + global projector)
// Mode: color replacing (incl. initial depth layout)
// ************************************************************************

// material switches

#define ENABLE_DIFFUSEMAP
#define ENABLE_SPECULARMAP
#define ENABLE_NORMALMAP
// ENABLE_DECALMAP
// ENABLE_AMBIENT_OCCLUSION_MAP
// ENABLE_DIRECTIONLESS_LIGHTING
// ENABLE_SKINNING
// ENABLE_REFLECTION_MAP
// ENABLE_SHADOWS
// ENABLE_ADDITIVE
// ENABLE_ALPHABLEND
// ENABLE_ALPHA_AS_COVERAGE
// ENABLE_DEBUG_MODE
// ENABLE_WATER
// MAX_LIGHTS
// SHADOW_QUALITY
// SHADER_QUALITY

// Prologue

#define _gamma(x) (sqrt(max(3.84e-6h,x)))
#define _degamma(x) (x*x)

#if PIXELSHADER_VERSION >= 30
	#define VS_PROFILE vs_3_0
	#define PS_PROFILE ps_3_0
#elif PIXELSHADER_VERSION >= 25
	#define VS_PROFILE vs_2_0
	#define PS_PROFILE ps_2_b
#endif

#define FRESNEL_CAP (1/32.h)

half4 TransparencyFunction( half4 color )
{
	float4 result = color;
#ifdef ENABLE_ALPHABLEND  
	result.rgb = color.rgb * color.a;
#endif
	return result;
}

#if SHADOW_QUALITY >= 2 && PIXELSHADER_VERSION < 30
	#undef SHADOW_QUALITY
	#define SHADOW_QUALITY 1
#endif


// ------------------------------------------------------------------------
// input registers
// ------------------------------------------------------------------------

#ifdef ENABLE_SKINNING
#define NTRANSFORMS 39
#else
#define NTRANSFORMS 3
#endif

// Transforms
// 0	ViewProjectionTM
// 1	ProjectorTM
// 2..	WorldTMs
float4x4	Transforms[ NTRANSFORMS ];
float4x4	World2CubeTM;
float4	View2World[2];
float4	CameraPos;			// w = inverse shadow map size
float4	UncompressData[3];	// 0 = mul, 1 = add, 2 = screen projection offset
float4	DebugData;			// x = mode, y = num lights
float4  ShadowProjParams;	// x = mul to objDist, y = add to objDist


// Material data
// 0	Diffuse color | Alpha
// 1	Specular color | Gloss
// 2	Incandescence
// 3    Rim color
float4		MaterialData[4];

// Environment data
// 0	Directional Direction
// 1	Directional Irradiance
// 2	Hemisphere base
// 3	Hemisphere grad
float4		EnvironmentData[4];

// Light data
// 0	Position (camera rel.) | Inverse outer range^2
// 1	Irradiance | Inner range ^2
// 2    Hemisphere axis | hemisphere enable
float4		LightData[ MAX_LIGHTS * 3 ];


// Fog data
// 0	Beta | StartLength
// 1	MinTransmittance | DirectAOInfluence
// 2	ExternalTransmittance 
// 3	Background
// 4	ExternalInScatter
float4		FogData[5];

// Water data
// 0	Color | Beta * Viewnear
// 1    Texcoord plane | Turbulence
// 2    Normalmap plane 1 | weight 1
// 3    Normalmap plane 2 | weight 2
// 4    Normalmap plane 3 | weight 3
float4		WaterData[5];

// ------------------------------------------------------------------------
// Texture Maps
// ------------------------------------------------------------------------

texture		DiffuseMap;
texture		SpecularMap;
texture		NormalMap;
texture     EnvironmentMap;
texture		DecalOrAmbientOcclusionMap;
texture		ShadowMap;
texture		DepthCaptureMap;
texture     FrameCaptureMap;

// ------------------------------------------------------------------------
// Sampler states
// ------------------------------------------------------------------------

sampler DiffuseMapSampler = sampler_state
{
   Texture = (DiffuseMap);
   MinFilter = MINFILTER1;
   MagFilter = Linear;
   MipFilter = MIPFILTER1;
   AddressU  = Wrap;
   AddressV  = Wrap;
   AddressW  = Wrap;
   MaxAnisotropy = 4;
};


sampler SpecularMapSampler = sampler_state
{
   Texture = (SpecularMap);
   MinFilter = MINFILTER2;
   MagFilter = Linear;
   MipFilter = MIPFILTER2;
   AddressU  = Wrap;
   AddressV  = Wrap;
   AddressW  = Wrap;
   MaxAnisotropy = 4;
};


sampler NormalMapSampler = sampler_state
{
   Texture = (NormalMap);
   MinFilter = MINFILTER1;
   MagFilter = Linear;
   MipFilter = MIPFILTER1;
   AddressU  = Wrap;
   AddressV  = Wrap;
   AddressW  = Wrap;
   MaxAnisotropy = 4;
};



sampler EnvironmentMapSampler = sampler_state
{
   Texture = (EnvironmentMap);
   MinFilter = MINFILTER2;
   MagFilter = Linear;
   MipFilter = MINFILTER2;
   AddressU  = Clamp;
   AddressV  = Clamp;
   AddressW  = Clamp;
   MaxAnisotropy = 2;
};



sampler DecalOrAmbientOcclusionMapSampler = sampler_state
{
   Texture = (DecalOrAmbientOcclusionMap);
   MinFilter = MINFILTER2;
   MagFilter = Linear;
   MipFilter = MIPFILTER2;
   AddressU  = Wrap;
   AddressV  = Wrap;
   AddressW  = Wrap;
   MaxAnisotropy = 2; 
};


sampler ShadowMapSampler = sampler_state
{
   Texture = (ShadowMap);
   MinFilter = Point;
   MagFilter = Point;
   MipFilter = None;
   AddressU  = Border;
   AddressV  = Border;
   AddressW  = Border;
   BorderColor = 0xFFFFFFFF;
   MaxAnisotropy = 2;  
};


sampler DepthCaptureMapSampler = sampler_state
{
   Texture = (DepthCaptureMap);
   MinFilter = Point;
   MagFilter = Point;
   MipFilter = Point;
   AddressU  = Clamp;
   AddressV  = Clamp;
   AddressW  = Clamp;
   MaxAnisotropy = 1;
};


sampler FrameCaptureMapSampler = sampler_state
{
   Texture = (FrameCaptureMap);
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;
   //AddressU  = Mirror;
   //AddressV  = Mirror;
   AddressU  = Clamp;
   AddressV  = Clamp;
   AddressW  = Clamp;
   MaxAnisotropy = 1;
};


// ------------------------------------------------------------------------
// Vertex- und Pixelshader inputs
// ------------------------------------------------------------------------


struct sVertexIn
{
    float4 P 		: POSITION0;
    half3 N 		: NORMAL0;
    float2 tc0 		: TEXCOORD0;
    float2 tc1 		: TEXCOORD1;
    half4 U			: COLOR0;
    half3 V			: COLOR1;

#ifdef ENABLE_SKINNING
    half3 weight	: BLENDWEIGHT;
    float4 index	: BLENDINDICES;
#endif
};


struct sPixelIn
{
    float4 Position					: POSITION;
	half3 FogTransmittance			: COLOR0;		
	half3 FogInScatter				: COLOR1;
    float4 Texcoord					: TEXCOORD0;	// texcoord (xy = set0, zw = set1)
    float4 TexcoordProj				: TEXCOORD1;    // projector coordinates
	float4 TexcoordScreen			: TEXCOORD2;    // screen projection coordinates
    float3 V						: TEXCOORD4;    // view vector
#ifndef ENABLE_DIRECTIONLESS_LIGHTING
    half3 N			       			: TEXCOORD5;	// normal
	half3 T							: TEXCOORD6;    // tangent
    half3 B							: TEXCOORD7;    // bi-tangent
#endif
};


// ------------------------------------------------------------------------
// Vertexshader: Realtime implementaion of the 3dsmax standard material
// ------------------------------------------------------------------------


sPixelIn VS_Default( sVertexIn IN )
{
	sPixelIn Out;

    Out.Texcoord.xy = IN.tc0;
    Out.Texcoord.zw = IN.tc1;

#ifdef ENABLE_SKINNING

    int4 convertedindices = D3DCOLORtoUBYTE4( IN.index ) + 2;

    half w4 = 1 - IN.weight[0] - IN.weight[1] - IN.weight[2];

    float4x4 Object2WorldTM =
    	Transforms[ convertedindices.x ] * IN.weight[0] +
    	Transforms[ convertedindices.y ] * IN.weight[1] +
    	Transforms[ convertedindices.z ] * IN.weight[2] +
    	Transforms[ convertedindices.w ] * w4;

#else

    float4x4 Object2WorldTM = Transforms[2];

#endif

	float4 objectP = IN.P * UncompressData[0] + UncompressData[1];

    float4 P = mul( objectP, Object2WorldTM );
    float3 N = mul( IN.N, Object2WorldTM );
    Out.Position = mul( P, Transforms[0] );

    float3 T = mul( 2 * IN.U.xyz - 1, Object2WorldTM );
    float3 B = mul( 2 * IN.V.xyz - 1, Object2WorldTM );
    float3 V = ( CameraPos - P ) * 0.01h;

	Out.V = V;

#ifndef ENABLE_DIRECTIONLESS_LIGHTING

	// The scale factor of the Object2WorldTM also affects the normal and tangent vectors.
	// In the case of compressed vertices, also the scale inherent in the Short2ObjTM.
    Out.N = N;
    Out.T = T;
    Out.B = B;

#endif

	// Fog
	half3 optical_length = FogData[0] * max( 0, length( V ) - FogData[0].w );
	half3 transmittance = max( FogData[1], exp2( - optical_length ) );
	half3 inscatter = FogData[3] * ( 1 - transmittance );
	Out.FogTransmittance.xyz = transmittance * FogData[2];
	Out.FogInScatter = sqrt( max( 3.84e-6h, inscatter * FogData[2] + FogData[4] ) );

	// Screen Projection
	Out.TexcoordScreen = Out.Position + Out.Position.w * UncompressData[2];

	// Shadow Map
	Out.TexcoordProj = mul( P, Transforms[1] );

    return Out;
}



// ------------------------------------------------------------------------
// Pixelshader: Realtime implementaion of the 3dsmax standard material
//              with an extra multi range lightmap
// ------------------------------------------------------------------------


float4 Fetch2DWithOffset( sampler2D S, float2 uv, float ofsx, float ofsy )
{
#ifdef _X36
	float4 result = 0;
	asm
	{
		tfetch2D result, uv, S, OffsetX = ofsx, OffsetY = ofsy
	};
	return result;
#else
	return tex2D( S, uv + CameraPos.w * float2( ofsx, ofsy ) );
#endif
}


half2 calculateAttenuation( float d2, float invr2, float k2 )
{
	// Inputs:
	// d2 = distance from the light source, squared
	// r2 = range of the light source, squared
	// k2 = inner range of the light source, squared
	
	half K = k2 / ( k2 + d2 );
	half D = d2 / ( k2 + d2 );
    half X = ( k2 + 1 ) / ( k2 + d2 );
	half R = saturate( 1 - d2 * invr2 );

	return half2( K, D ) * X * R * R;
}


half calculateAttenuationDirectionless( float d2, float invr2, float k2 )
{
	// Inputs:
	// d2 = distance from the light source, squared
	// r2 = range of the light source, squared
	// k2 = inner range of the light source, squared

	half X = ( k2 + 1 ) / ( k2 + d2 );
	half R = saturate( 1 - d2 * invr2 );
	return X * R * R;
}


half3 calculateAmbientSpecularEnv( half3 R, half4 scg, half ao, half invdotNV )
{
	// Take the specular contribution from the a miplevel of the environment map in the direction of the reflection vector.
	// The miplevel is determined by the norm of the reflection vector differential.
	// However, the glossiness determines a lower bound on the differential (lower glossiness -> higher miplevel).

#if SHADER_QUALITY >= 2
	half g = scg.a;
	half3 R_cube = mul( R, World2CubeTM );
	half4 lodfactors = World2CubeTM[3];	
	half rlod = ( lodfactors.w * g + lodfactors.z ) * g + lodfactors.y;
	half fresnel = pow( invdotNV * g, 4 );
	half3 speccolor = scg.xyz + fresnel * ( saturate( 64 * scg.xyz ) - scg.xyz );
	half3 t1 = ao * World2CubeTM[2].w;
	half3 t2 = speccolor * _degamma( texCUBElod( EnvironmentMapSampler, half4( R_cube, rlod ) ).xyz );
	return t1 * t2;
#elif SHADER_QUALITY >= 1
	half g = scg.a;
	half3 R_cube = mul( R, World2CubeTM );
	half4 lodfactors = World2CubeTM[3];	
	half rlod = ( lodfactors.w * g + lodfactors.z ) * g + lodfactors.y;
	half3 speccolor = scg.xyz;
	half3 t1 = ao * World2CubeTM[2].w;
	half3 t2 = speccolor * _degamma( texCUBElod( EnvironmentMapSampler, half4( R_cube, rlod ) ).xyz );
	return t1 * t2;
#else
	half3 R_cube = mul( R, World2CubeTM );
	half3 speccolor = scg.xyz;
	half3 t1 = ao * World2CubeTM[2].w;
	half3 t2 = speccolor * _degamma( texCUBE( EnvironmentMapSampler, R_cube ) );
	return t1 * t2;
#endif
}


half3 calculateAmbientSpecularCol( half3 hbase, half3 hgrad, half3 R, half4 scg, half ao, half invdotNV )
{
	// Take the specular contribution from the environment colors.

#if SHADER_QUALITY >= 2
	half g = scg.a;
	half q = .25h / ( 1 - 0.995h * g );
	half fresnel = pow( invdotNV * g, 4 );
	half3 speccolor = scg.xyz + fresnel * ( saturate( 64 * scg.xyz ) - scg.xyz );
	half3 t1 = ao * speccolor;
	return t1 * ( hbase + ( saturate( R.z * q + q ) * 2 - 1 ) * hgrad );
#elif SHADER_QUALITY >= 1
	half g = scg.a;
	half q = .25h / ( 1 - 0.995h * g );
	half3 speccolor = scg.xyz;
	half3 t1 = ao * speccolor;
	return t1 * ( hbase + ( saturate( R.z * q + q ) * 2 - 1 ) * hgrad );
#else
	half q = 4;
	half3 speccolor = scg.xyz;
	half3 t1 = ao * speccolor;
	return t1 * ( hbase + ( saturate( R.z * q + q ) * 2 - 1 ) * hgrad );
#endif
}


half4 PS_Default( sPixelIn IN ) : COLOR
{

#ifdef ENABLE_WATER
	IN.Texcoord.xy = WaterData[1].xy + IN.Texcoord.xy * WaterData[1].z;
#endif

	// ........................................................................
	// normal map
	// ........................................................................

#ifdef ENABLE_DIRECTIONLESS_LIGHTING
	half3 N = 0;
	half dotNL = 1;
	half invdotNV = 0;

#else
	half3 N = IN.N;
	half3 normalMap = 0;

  #ifdef ENABLE_NORMALMAP
    
    #ifdef ENABLE_WATER
	half3 watersurf2 = tex2D( NormalMapSampler, WaterData[2].xy + IN.Texcoord.xy * WaterData[2].z );
	half3 watersurf3 = tex2D( NormalMapSampler, WaterData[3].xy + IN.Texcoord.xy * WaterData[3].z );
	half3 watersurf4 = tex2D( NormalMapSampler, WaterData[4].xy + IN.Texcoord.xy * WaterData[4].z );
	#else
	normalMap = tex2D( NormalMapSampler, IN.Texcoord );
	#endif
	
	#ifdef NORMALMAP_UNSIGNED
	  #ifdef ENABLE_WATER
	  watersurf2.xy = 2 * watersurf2.xy - 1;
	  watersurf3.xy = 2 * watersurf3.xy - 1;
	  watersurf4.xy = 2 * watersurf4.xy - 1;
	  #else
	  normalMap.xy = 2 * normalMap.xy - 1;
	  #endif
	#endif
	
	#ifdef ENABLE_WATER
	normalMap = watersurf2 * WaterData[2].w + 
	            watersurf3 * WaterData[3].w + 
	            watersurf4 * WaterData[4].w;
	#endif

	normalMap.z = sqrt( 1 - 0.9995h * dot( normalMap.xy, normalMap.xy ) );
	
	half3x3 Tangent2WorldTM = half3x3( IN.T, IN.B, IN.N );
	half3 N_linear = mul( normalMap, Tangent2WorldTM );
  #endif

	N = normalize( N_linear );
	half3 L = EnvironmentData[0];
	half3 V = normalize( IN.V );
	half3 H = normalize( L + V );
	half3 R = reflect( -V, N );
	
	half dotNL = saturate( dot( N, L ) );
	float dotNH = saturate( dot( N, H ) );
	half dotNV = saturate( dot( N, V ) );
	half invdotNV = saturate( 1 - dotNV );
	half dotLH = saturate( dot( L, H ) );
	half invdotLH = saturate( 1 - dotLH );
#endif

#ifdef ENABLE_WATER
	IN.Texcoord.xy += normalMap.xy * WaterData[1].w;
#endif

	// ........................................................................
    // diffuse color & opacity
    // ........................................................................

    half4 diffuse_color_alpha = MaterialData[0];

#ifdef ENABLE_DIFFUSEMAP	
	half4 diffuse_map = tex2D( DiffuseMapSampler, IN.Texcoord );
	diffuse_map.xyz = _degamma( diffuse_map.xyz );
	diffuse_color_alpha *= diffuse_map;
#endif

	// ........................................................................
	// specular color & glossiness
	// ........................................................................

	half4 specular_color_gloss = MaterialData[1];

#ifdef ENABLE_SPECULARMAP
	half4 specular_map = tex2D( SpecularMapSampler, IN.Texcoord );
	specular_map.xyz = _degamma( specular_map.xyz );
	specular_color_gloss *= specular_map;
#endif

	// limit glossiness to surface curvature

#ifndef ENABLE_DIRECTIONLESS_LIGHTING
#if SHADER_QUALITY >= 1
	{
		float3 ddxN = ddx(N);
		float3 ddyN = ddy(N);
		float curv2 = max( dot( ddxN, ddxN ), dot( ddyN, ddyN ) );
		float gloss_max = - 0.0909 - 0.0909 * log2( 0.5 * curv2 );
		
		specular_color_gloss.a = min( specular_color_gloss.a, gloss_max );
	}
#endif
#endif

	half specular_exponent = exp2( 1 + specular_color_gloss.a * 11 );

#ifdef ENABLE_DEBUG_MODE
	if( DebugData.x == 19 )
		specular_color_gloss.xyz = 1;
#endif

	// ........................................................................
	// decal/AO map
	// ........................................................................

	half ambient_occlusion = 1;

#ifdef ENABLE_DECALMAP
	half4 decal_map = tex2D( DecalOrAmbientOcclusionMapSampler, IN.Texcoord.zw );
	decal_map.xyz = _degamma( decal_map.xyz );
	diffuse_color_alpha.xyz = lerp( diffuse_color_alpha, decal_map, decal_map.a );
#endif
	
#ifdef ENABLE_AMBIENT_OCCLUSION_MAP 
	ambient_occlusion = tex2D( DecalOrAmbientOcclusionMapSampler, IN.Texcoord.zw ).a;
	ambient_occlusion = _degamma( ambient_occlusion );
#endif

	// ........................................................................
	// shadow map
	// ........................................................................

	half shadow_test = 1;

#if SHADOW_QUALITY >= 0 && defined( ENABLE_SHADOWS )
#if SHADOW_QUALITY >= 2
	{
		// BICUBIC
		float2 uv = IN.TexcoordProj.xy / IN.TexcoordProj.w;
		
	#ifdef _X36
		// workaround for possible shader compiler bug
		float z = max( 0, -IN.TexcoordProj.z );
	#else
		float z = -IN.TexcoordProj.z;
	#endif

		float4 D[4] = 
		{
			float4( 
				Fetch2DWithOffset( ShadowMapSampler, uv, -1.5, -1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, -0.5, -1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +0.5, -1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +1.5, -1.5 ).x ),
			float4( 
				Fetch2DWithOffset( ShadowMapSampler, uv, -1.5, -0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, -0.5, -0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +0.5, -0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +1.5, -0.5 ).x ),
			float4( 
				Fetch2DWithOffset( ShadowMapSampler, uv, -1.5, +0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, -0.5, +0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +0.5, +0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +1.5, +0.5 ).x ),
			float4( 
				Fetch2DWithOffset( ShadowMapSampler, uv, -1.5, +1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, -0.5, +1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +0.5, +1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +1.5, +1.5 ).x )
		};
		
		half4 Q = half4( 1, 0, 1, 0 ) + half4( -1, +1, -1, +1 ) * frac( uv / CameraPos.w - .5 ).xxyy;
		Q = Q * Q;
		half4 WX = half4( 0, .5, .5, 0 ) + half4( +.25, -.25, -.25, +.25 ) * Q.xyxy;
		half4 WY = half4( 0, .5, .5, 0 ) + half4( +.25, -.25, -.25, +.25 ) * Q.zwzw;

		shadow_test = dot( half4( 
			dot( D[0] > z, WX ),
			dot( D[1] > z, WX ),
			dot( D[2] > z, WX ),
			dot( D[3] > z, WX ) ), WY );
	}

#elif SHADOW_QUALITY >= 1

	{
		// BILINEAR
		float2 uv = IN.TexcoordProj.xy / IN.TexcoordProj.w;
		float z = -IN.TexcoordProj.z;

		float4 D = float4( 
			Fetch2DWithOffset( ShadowMapSampler, uv, -.5, -.5 ).x,
			Fetch2DWithOffset( ShadowMapSampler, uv, +.5, -.5 ).x,
			Fetch2DWithOffset( ShadowMapSampler, uv, -.5, +.5 ).x,
			Fetch2DWithOffset( ShadowMapSampler, uv, +.5, +.5 ).x );

		half4 Q = half4( 1, 0, 1, 0 ) + half4( -1, +1, -1, +1 ) * frac( uv / CameraPos.w - .5 ).xxyy;
		half4 W = Q.zzww * Q.xyxy;

		shadow_test = dot( D > z, W );
	}

#endif // SHADOW_QUALITY 

	{
		float objDist = length( IN.V );
		float blendValue = saturate( objDist * ShadowProjParams.x + ShadowProjParams.y  );
		shadow_test = lerp( shadow_test, 1, blendValue );
	}

#endif // SHADOW_QUALITY >= 0 && defined( ENABLE_SHADOWS )				
	
		
	// ........................................................................
	// sum up all lights components
	// ........................................................................

	half3 kDiffuse = 0;
	half3 kSpecular = 0;

	// calculate ambient component

	{
		half3 hbase = EnvironmentData[2];
		half3 hgrad = EnvironmentData[3];

#ifdef ENABLE_ENVIRONMENT_MAP		
		// If an environment map was defined, we'll use this to calculate the ambient lighting.
		// Look up the diffuse envmap in the normal direction.
		// Look up the specular envmap in the direction of the reflection vector.

  #ifdef ENABLE_DIRECTIONLESS_LIGHTING  
        kDiffuse += ambient_occlusion * hbase;
  #else
		kDiffuse += ambient_occlusion * ( hbase + N.z * hgrad );
		kSpecular += calculateAmbientSpecularEnv( R, specular_color_gloss, ambient_occlusion, invdotNV );
  #endif

#else
		// Otherwise use the sky- & ground color (hemisphere lighting).
  #ifdef ENABLE_DIRECTIONLESS_LIGHTING
		kDiffuse += ambient_occlusion * hbase;
  #else
		// Take the diffuse contribution from the ambient colors.		
		kDiffuse += ambient_occlusion * ( hbase + N.z * hgrad );
		kSpecular += calculateAmbientSpecularCol( hbase, hgrad, R, specular_color_gloss, ambient_occlusion, invdotNV );
  #endif

#endif // ENVMAP
	}


	// attenuate the ambient with the shadow, if water
#ifdef ENABLE_WATER
	kDiffuse = lerp( shadow_test * kDiffuse, kDiffuse, .2 );
	kSpecular = lerp( shadow_test * kSpecular, kSpecular, .2 );
#endif

	// calculate directional component

	{
		half dao = saturate( lerp( 1, ambient_occlusion, FogData[1].w ) );

		half3 light_color = EnvironmentData[1] * shadow_test;

#ifdef ENABLE_DIRECTIONLESS_LIGHTING
		kDiffuse += light_color;
#else	
		half D = .5h * ( specular_exponent + 1 ) * pow( dotNH, specular_exponent );
#if SHADER_QUALITY >= 1
		half FG = .25h / ( pow( dotLH, 3 ) + FRESNEL_CAP );
#else
		half FG = .25h;
#endif
		half DFG = D * FG;
		kDiffuse += dao * light_color * dotNL;
		kSpecular += dao * light_color * dotNL * DFG * specular_color_gloss.xyz;
#endif
	}


	// calculate effect from secondary lights

#if MAX_LIGHTS > 0
	for( int i = 0; i < DebugData.y; ++i )
	{
		half4 dataA = LightData[3*i+0];
		half4 dataB = LightData[3*i+1];
		half4 dataC = LightData[3*i+2];

		half3 dx = dataA.xyz + IN.V;
		half d2 = dot( dx, dx );
		half3 L = normalize( dx );
		half3 lightcolor = dataB.xyz;

#ifdef ENABLE_DIRECTIONLESS_LIGHTING
		half attenuation = calculateAttenuationDirectionless( d2, dataA.w, dataB.w );
		kDiffuse += lightcolor * attenuation;
#else
		half2 attenuation = calculateAttenuation( d2, dataA.w, dataB.w );

		half3 H = normalize( L + V );
		half dotNL = saturate( dot( N, L ) );
		float dotNH = saturate( dot( N, H ) );

		half dao = saturate( lerp( 1, ambient_occlusion, FogData[1].w ) );

		half backfactor1 = dataC.w ? saturate( dot( dx, dataC.xyz ) ) : 1;
		half backfactor2 = dataC.w ? saturate( .5 + .5 * dot( N, dataC.xyz ) ) : 1;
		half2 diffatt = half2( ambient_occlusion * backfactor2, dao * dotNL ) * backfactor1;
		half2 specatt = diffatt;

		half D = .5h * ( specular_exponent + 1 ) * pow( dotNH, specular_exponent );
#if SHADER_QUALITY >= 1
		half dotLH = saturate( dot( L, H ) );
		half FG = .25h / ( pow( dotLH, 3 ) + FRESNEL_CAP );
#else
		half FG = .25h;
#endif
		half DFG = D * FG;
		specatt.y *= DFG;
		kDiffuse += lightcolor * dot( attenuation, diffatt );
		kSpecular += specular_color_gloss.xyz * lightcolor * dot( attenuation, specatt );
#endif
	}
#endif // MAX_SECONDARY_LIGHTS > 0

	// color sum

	half opacity = 1;
	half linopacity = 1;

#ifdef ENABLE_WATER

    float3 screenpos = IN.TexcoordScreen.xyz / IN.TexcoordScreen.w;
	float2 screentexcoord = screenpos.xy * float2( .5, -.5 ) + float2( .5, .5 );
		
	float Zp = screenpos.z;
	float Zm = tex2D( DepthCaptureMapSampler, screentexcoord ).x;
		
	float Dp = WaterData[0].w / ( 1 - Zp );
	float Dm = WaterData[0].w / ( 1 - Zm );

	half3 dV = V - normalize( V + 0.01h * N );
	float2 dp = float2( dot( dV, View2World[0] ), 0 );
	
	float Zm2 = tex2D( DepthCaptureMapSampler, screentexcoord + dp ).x;
	Dm = Zm2 > Zp ? WaterData[0].w / ( 1 - Zm2 ) : Dm;
	dp = Zm2 > Zp ? dp : 0;

	half T0 = saturate( exp2( Dp - Dm ) );
	half T1 = 1 - pow( diffuse_color_alpha.a, 2 );	
	half Tfresnel = 1 - pow( invdotNV * specular_color_gloss.a, 4 );

	half3 background = _degamma( tex2D( FrameCaptureMapSampler, screentexcoord + dp ).xyz );
	half3 I0 = WaterData[0] * kDiffuse;
	half3 I1 = diffuse_color_alpha * kDiffuse;

	half3 col;
	col = lerp( I0, background, T0 );
	col = lerp( I1, col, T1 );
	col = kSpecular + col * Tfresnel;
		
#else

  #ifdef ENABLE_ALPHABLEND

    #ifdef ENABLE_ALPHA_AS_COVERAGE
	opacity = diffuse_color_alpha.a;
	#else
	opacity = pow( diffuse_color_alpha.a, 1 - pow( invdotNV, 2 ) );
	#endif

	linopacity = opacity * opacity;

  #endif

	half3 col = linopacity * diffuse_color_alpha.xyz * ( MaterialData[2] + kDiffuse );

  #ifdef ENABLE_ALPHA_AS_COVERAGE 
	col += linopacity * kSpecular;
  #else
	col += kSpecular;
  #endif

	// rim color effect

  #ifndef ENABLE_DIRECTIONLESS_LIGHTING
	col += linopacity * pow( invdotNV, 4 ) * MaterialData[3];
  #endif

#endif

	// ........................................................................
	// Fog
	// ........................................................................

#ifdef ENABLE_ADDITIVE
	col = col * IN.FogTransmittance;
#else
	col = col * IN.FogTransmittance + IN.FogInScatter * IN.FogInScatter * linopacity;
#endif

	col *= FogData[2].w;

	// ........................................................................
	// Return final result
	// ........................................................................

	half4 result;
	result.rgb = saturate( _gamma( col ) );

#ifdef ENABLE_ALPHABLEND
	result.a = opacity;
#else
	result.a = diffuse_color_alpha.a;
#endif

#ifdef ENABLE_DEBUG_MODE

	float DebugMode = DebugData.x;
	float NumLights = DebugData.y;

	if( DebugMode > 0 )
	{
		switch( DebugMode )
		{
		case 1:
			result.rgb = _gamma( diffuse_color_alpha.rgb );
			break;

		case 2:
			result.rgb = diffuse_color_alpha.a;
			break;

		case 3:
			result.rgb = result.a;
			break;

		case 4:
			result.rgb = _gamma( specular_color_gloss.rgb );
			break;

		case 5:
			result.rgb = specular_color_gloss.a;
			break;

		case 6:
			result.rgb = _gamma( ambient_occlusion );
			break;

		case 7:
		#ifndef ENABLE_DIRECTIONLESS_LIGHTING
			result.rgb = normalize( IN.N ) * .5 + .5;
		#else
			result.rgb = .5;
		#endif
			break;

		case 8:
		#ifndef ENABLE_DIRECTIONLESS_LIGHTING
			result.rgb = normalize( N ) * .5 + .5;
		#else
			result.rgb = .5;
		#endif
			break;

		case 9:
		#ifndef ENABLE_DIRECTIONLESS_LIGHTING
			result.rgb = normalize( IN.T ) * .5 + .5;
		#else
			result.rgb = .5;
		#endif
			break;
		
		case 10:
		#ifndef ENABLE_DIRECTIONLESS_LIGHTING
			result.rgb = normalize( IN.B ) * .5 + .5;
		#else
			result.rgb = .5;
		#endif
			break;

		case 11:
			result.rgb = half3( IN.Texcoord.x, 0, IN.Texcoord.y );
			break;

		case 12:
			result.rgb = half3( IN.Texcoord.z, 0, IN.Texcoord.w );
			break;

		case 13:
			result.rgb = IN.FogTransmittance;
			break;

		case 14:
			result.rgb = IN.FogInScatter;
			break;

		case 15:
			result.rgb = normalize( IN.V ) * .5 + .5;
			break;

		case 16:
			switch( NumLights )
			{
			case 0: result.rgb = half3( 0.1, 0.03, 0.01 ); break;
			case 1: result.rgb = half3( 0.2, 0.08, 0.03 ); break;
			case 2: result.rgb = half3( 0.3, 0.2, 0.1 ); break;
			case 3: result.rgb = half3( 0.4, 0.3, 0.3 ); break;
			case 4: result.rgb = half3( 0.5, 0.45, 0.5 ); break;
			case 5: result.rgb = half3( 0.6, 0.6, 0.7 ); break;
			case 6: result.rgb = half3( 0.7, 0.85, 0.9 ); break;
			case 7: result.rgb = half3( 0.8, 0.9, 0.95 ); break;
			}
			break;

		case 17:
			result.rgb = half3( 
				_gamma( max( max( specular_color_gloss.r, specular_color_gloss.g ), specular_color_gloss.b ) ),
				0,
				specular_color_gloss.a );
			break;

		case 18:
			result.rgb = _gamma( kDiffuse );
			break;

		case 19:
			result.rgb = _gamma( kSpecular );
			break;

		case 20:
			result.rgb = _gamma( MaterialData[2] + MaterialData[3] * pow( invdotNV, 4 ) );
			break;
		}
	
		result.a = 1;
	}

#endif // Debug mode

	return result;
}



// ------------------------------------------------------------------------
// Primary Render effect
// ------------------------------------------------------------------------


technique t0
{
	pass p0
	{
        VertexShader = compile VS_PROFILE VS_Default(); 
        PixelShader = compile PS_PROFILE PS_Default();
	}
}


