
// ************************************************************************
// Replay FX file for world shading
// Pass: primary sphere lights 
// Mode: color accumulating
// There can be any number of passes, 1 pass for each primary light
// ************************************************************************

// switches

#define ENABLE_DIFFUSEMAP
#define ENABLE_SPECULARMAP
#define ENABLE_NORMALMAP						
// ENABLE_DECALMAP
// ENABLE_AMBIENT_OCCLUSION_MAP
// ENABLE_DIRECTIONLESS_LIGHTING
// ENABLE_SKINNING
// ENABLE_SHADOWS
// ENABLE_ALPHABLEND
// ENABLE_ALPHA_AS_COVERAGE
// ENABLE_WATER


// LIGHT_TYPE 0 = directional, 1 = spot

// Prologue

#define _gamma(x) (sqrt(max(3.84e-6h,x)))
#define _degamma(x) (x*x)

#if PIXELSHADER_VERSION >= 30
	#define VS_PROFILE vs_3_0
	#define PS_PROFILE ps_3_0
#elif PIXELSHADER_VERSION >= 25
	#define VS_PROFILE vs_2_0
	#define PS_PROFILE ps_2_b
#endif

#define FRESNEL_CAP (1/32.h)

float4 TransparencyFunction( float4 color )
{
	float4 result = color;
#ifdef ENABLE_ALPHABLEND  
	result.rgb = color.rgb * color.a;
#endif
	return result;
}

// ------------------------------------------------------------------------
// input registers
// ------------------------------------------------------------------------

#ifdef ENABLE_SKINNING
#define NTRANSFORMS 39
#else
#define NTRANSFORMS 3
#endif

// Transforms
// 0	ViewProjectionTM
// 1	ProjectorTM
// 2..	WorldTMs
float4x4	Transforms[ NTRANSFORMS ];
float4		View2World[2];
float4		CameraPos; // w = inverse shadow map size
float4		UncompressData[3];	// 0 = mul, 1 = add, 2 = viewport
float4      DebugData;			// x = mode, y = num lights


// Material data
// 0	Diffuse color | Alpha
// 1	Specular color | Gloss
float4		MaterialData[2];

// Light data
// 0	Position (camera rel.) | inverse outer range^2
// 1	Irradiance 
float4		LightData[3];

// Fog data
// 0	Beta / StartLength
// 1	MinTransmittance / DirectAOInfluence
// 2	ExternalTransmittance
float4		FogData[3];

// Water data
// 0	Color | Beta * Viewnear
// 1    Texcoord plane
// 2    Normalmap plane 1 | weight 1
// 3    Normalmap plane 2 | weight 2
// 4    Normalmap plane 3 | weight 3
float4		WaterData[5];


// ------------------------------------------------------------------------
// Texture Maps
// ------------------------------------------------------------------------

texture		DiffuseMap;
texture		SpecularMap;
texture		NormalMap;
texture		DecalOrAmbientOcclusionMap;
texture		ProjectorMap;
texture		ShadowMap;
texture		DepthCaptureMap;

// ------------------------------------------------------------------------
// Sampler states
// ------------------------------------------------------------------------

sampler DiffuseMapSampler = sampler_state
{
   Texture = (DiffuseMap);
   MinFilter = MINFILTER1;
   MagFilter = Linear;
   MipFilter = MIPFILTER1;
   AddressU  = Wrap;
   AddressV  = Wrap;
   AddressW  = Wrap;
   MaxAnisotropy = 4;
};


sampler SpecularMapSampler = sampler_state
{
   Texture = (SpecularMap);
   MinFilter = MINFILTER2;
   MagFilter = Linear;
   MipFilter = MIPFILTER2;
   AddressU  = Wrap;
   AddressV  = Wrap;
   AddressW  = Wrap;
   MaxAnisotropy = 4;
};


sampler NormalMapSampler = sampler_state
{
   Texture = (NormalMap);
   MinFilter = MINFILTER1;
   MagFilter = Linear;
   MipFilter = MIPFILTER1;
   AddressU  = Wrap;
   AddressV  = Wrap;
   AddressW  = Wrap;
   MaxAnisotropy = 4;
};


sampler DecalOrAmbientOcclusionMapSampler = sampler_state
{
   Texture = (DecalOrAmbientOcclusionMap);
   MinFilter = MINFILTER2;
   MagFilter = Linear;
   MipFilter = MIPFILTER2;
   AddressU  = Wrap;
   AddressV  = Wrap;
   AddressW  = Wrap;
   MaxAnisotropy = 2;
};


sampler ProjectorMapSampler = sampler_state
{
   Texture = (ProjectorMap);
   MinFilter = MINFILTER2;
   MagFilter = Linear;
   MipFilter = MIPFILTER2;
   AddressU  = Border;
   AddressV  = Border;
   AddressW  = Border;
   BorderColor = 0x00000000;
   MaxAnisotropy = 2;
};


sampler ShadowMapSampler = sampler_state
{
   Texture = (ShadowMap);
   MinFilter = Point;
   MagFilter = Point;
   MipFilter = None;
   AddressU  = Clamp;
   AddressV  = Clamp;
   AddressW  = Clamp;
   MaxAnisotropy = 2;
};



sampler DepthCaptureMapSampler = sampler_state
{
   Texture = (DepthCaptureMap);
   MinFilter = Point;
   MagFilter = Point;
   MipFilter = Point;
   AddressU  = Clamp;
   AddressV  = Clamp;
   AddressW  = Clamp;
   MaxAnisotropy = 1;
};



// ------------------------------------------------------------------------
// Vertex- und Pixelshader inputs
// ------------------------------------------------------------------------


struct sVertexIn
{
    float4 P 		: POSITION0;
    half3 N 		: NORMAL0;
    float2 tc0 		: TEXCOORD0;
    float2 tc1 		: TEXCOORD1;
    half3 U			: COLOR0;
    half3 V			: COLOR1;

#ifdef ENABLE_SKINNING
    half3 weight	: BLENDWEIGHT;
    float4 index	: BLENDINDICES;
#endif
};


struct sPixelIn
{
    float4 Position			: POSITION;
	half3 FogTransmittance	: COLOR0;		
    float4 Texcoord			: TEXCOORD0;	// texcoord (xy = set0, zw = set1)
    float4 TexcoordProj		: TEXCOORD1;    // projector texcoord
	float4 TexcoordScreen	: TEXCOORD2;    // screen projection coordinates
    half3 V                 : TEXCOORD3;    // view vector
#ifndef ENABLE_DIRECTIONLESS_LIGHTING
	half3 N            		: TEXCOORD4;	// normal
    half3 T                 : TEXCOORD5;    // tangent
    half3 B					: TEXCOORD6;    // bitangen
#endif
};


// ------------------------------------------------------------------------
// Vertexshader: Realtime implementaion of the 3dsmax standard material
// ------------------------------------------------------------------------

sPixelIn VS_Default( sVertexIn IN )
{
    sPixelIn Out;

    Out.Texcoord.xy = IN.tc0;
    Out.Texcoord.zw = IN.tc1;

#ifdef ENABLE_SKINNING

    int4 convertedindices = D3DCOLORtoUBYTE4( IN.index ) + 2;

    half w4 = 1 - IN.weight[0] - IN.weight[1] - IN.weight[2];

    float4x4 Object2WorldTM =
    	Transforms[ convertedindices.x ] * IN.weight[0] +
    	Transforms[ convertedindices.y ] * IN.weight[1] +
    	Transforms[ convertedindices.z ] * IN.weight[2] +
    	Transforms[ convertedindices.w ] * w4;

#else

    float4x4 Object2WorldTM = Transforms[2];

#endif

	float4 objectP = IN.P * UncompressData[0] + UncompressData[1];

    float4 P	 = mul( objectP, Object2WorldTM );
    half3 N      = mul( IN.N, Object2WorldTM );
    Out.Position = mul( P, Transforms[0] );

    half3 T = mul( 2 * IN.U - 1, Object2WorldTM );
    half3 B = mul( 2 * IN.V - 1, Object2WorldTM );
    half3 V = ( CameraPos.xyz - P ) * 0.01h;

#ifdef ENABLE_DIRECTIONLESS_LIGHTING

    Out.V = V;

#else

    Out.N = N;
    Out.T = T;
    Out.B = B;
    Out.V = V;

#endif

	// Fog
	half3 optical_length = FogData[0] * max( 0, length( V ) - FogData[0].w );
	half3 transmittance = max( FogData[1], exp2( - optical_length ) );
	Out.FogTransmittance.xyz = transmittance * FogData[2];

	Out.TexcoordProj = mul( P, Transforms[1] ); 

	// Screen Projection
	Out.TexcoordScreen = Out.Position + Out.Position.w * UncompressData[2];

    return Out;
}

// ------------------------------------------------------------------------
// Pixelshader: Realtime implementaion of the 3dsmax standard material
//              with an extra multi range lightmap
// ------------------------------------------------------------------------



float4 Fetch2DWithOffset( sampler2D S, float2 uv, float ofsx, float ofsy )
{
#ifdef _X36
	float4 result = 0;
	asm
	{
		tfetch2D result, uv, S, OffsetX = ofsx, OffsetY = ofsy
	};
	return result;
#else
	return tex2D( S, uv + CameraPos.w * float2( ofsx, ofsy ) );
#endif
}


half4 PS_Default( sPixelIn IN ) : COLOR
{
	// ........................................................................
	// projector map
	// ........................................................................

	half3 projector_color = 1;

	{
		// the spotlight cone
#if LIGHT_TYPE == 0
		float2 p = 2 * IN.TexcoordProj.xy / IN.TexcoordProj.w - 1;
		p *= p;
		projector_color = saturate( 2 - 2 * max( p.x, p.y ) );
#elif LIGHT_TYPE == 1
		float2 p = 2 * IN.TexcoordProj.xy / IN.TexcoordProj.w - 1;
		projector_color = pow( saturate( 1 - dot( p, p ) ), 2 );
#endif

		// backprojection clip
		projector_color *= ( IN.TexcoordProj.z < 0 );
		
		// the projector color (or white.dds)
		half3 projector_map = tex2Dproj( ProjectorMapSampler, IN.TexcoordProj );
		projector_map = _degamma( projector_map.xyz );
		projector_color *= projector_map;
	}

	if( dot( projector_color, projector_color ) <= 0 )
	{
		// If the projector color is black, the pixel is outside the projector.
		// We can skip the rest of the pixel shader.
		// But we do not use the clip instruction, because the clip instr would
		// execute the shader to its end IGNORING all further commands (including the return statement)
		// So in this case, not executing the rest of the shader saves more time than not drawing a completely transparent pixel.
		// TODO: setup an alphatest instead of clip?

		// clip(-1);
		return 0;
	}

#ifdef ENABLE_WATER
	IN.Texcoord.xy = WaterData[1].xy + IN.Texcoord.xy * WaterData[1].z;
#endif

	// ........................................................................
	// normal map
	// ........................................................................

#ifndef ENABLE_DIRECTIONLESS_LIGHTING
	half3 N = IN.N;
	half3 normalMap = 0;

  #ifdef ENABLE_NORMALMAP

	#ifdef ENABLE_WATER
	half3 watersurf2 = tex2D( NormalMapSampler, WaterData[2].xy + IN.Texcoord.xy * WaterData[2].z );
	half3 watersurf3 = tex2D( NormalMapSampler, WaterData[3].xy + IN.Texcoord.xy * WaterData[3].z );
	half3 watersurf4 = tex2D( NormalMapSampler, WaterData[4].xy + IN.Texcoord.xy * WaterData[4].z );
	#else
	normalMap = tex2D( NormalMapSampler, IN.Texcoord );
	#endif
		
	#ifdef NORMALMAP_UNSIGNED
	  #ifdef ENABLE_WATER
	  watersurf2.xy = 2 * watersurf2.xy - 1;
	  watersurf3.xy = 2 * watersurf3.xy - 1;
	  watersurf4.xy = 2 * watersurf4.xy - 1;
	  #else
	  normalMap.xy = 2 * normalMap.xy - 1;
	  #endif
	#endif
	
	#ifdef ENABLE_WATER
	normalMap = watersurf2 * WaterData[2].w + 
	            watersurf3 * WaterData[3].w + 
	            watersurf4 * WaterData[4].w;
	#endif

	normalMap.z = sqrt( 1 - 0.9995h * dot( normalMap.xy, normalMap.xy ) );
	half3x3 Tangent2WorldTM = half3x3( IN.T, IN.B, IN.N );
	half3 N_linear = mul( normalMap, Tangent2WorldTM );
  #endif

	N = normalize( N_linear );
	half3 V = normalize( IN.V );
	half invdotNV = saturate( 1 - dot( N, V ) );
#else
	half3 V = normalize( IN.V );
	half invdotNV = 0;
#endif

#ifdef ENABLE_WATER
	IN.Texcoord.xy += normalMap.xy * WaterData[1].w;
#endif

    // ........................................................................
    // diffuse color & opacity
    // ........................................................................

    half4 diffuse_color_alpha = MaterialData[0];

#ifdef ENABLE_DIFFUSEMAP
	half4 diffuse_map = tex2D( DiffuseMapSampler, IN.Texcoord );
	diffuse_map.xyz = _degamma( diffuse_map.xyz );
	diffuse_color_alpha *= diffuse_map;
#endif

	// ........................................................................
	// specular color & glossiness
	// ........................................................................

	half4 specular_color_gloss = MaterialData[1];

#ifdef ENABLE_SPECULARMAP
	half4 specular_map = tex2D( SpecularMapSampler, IN.Texcoord );
	specular_map.xyz = _degamma( specular_map.xyz );
	specular_color_gloss *= specular_map;
#endif

	// limit glossiness to surface curvature

#ifndef ENABLE_DIRECTIONLESS_LIGHTING
#if SHADER_QUALITY >= 1
	{		
		float3 ddxN = ddx(N);
		float3 ddyN = ddy(N);
		float curv2 = max( dot( ddxN, ddxN ), dot( ddyN, ddyN ) );
		float gloss_max = - 0.0909 - 0.0909 * log2( 0.5 * curv2 );
		
		specular_color_gloss.a = min( specular_color_gloss.a, gloss_max );
	}
#endif
#endif

	half specular_exponent = exp2( 1 + specular_color_gloss.a * 11 );

#ifdef ENABLE_DEBUG_MODE
	if( DebugData.x == 19 )
		specular_color_gloss.xyz = 1;
#endif

	// ........................................................................
	// decal/AO map
	// ........................................................................

	half ambient_occlusion = 1;

#ifdef ENABLE_DECALMAP
	half4 decal_map = tex2D( DecalOrAmbientOcclusionMapSampler, IN.Texcoord.zw );
	decal_map.xyz = _degamma( decal_map.xyz );
	diffuse_color_alpha.xyz = lerp( diffuse_color_alpha, decal_map, decal_map.a );
#endif
	
#ifdef ENABLE_AMBIENT_OCCLUSION_MAP 
	ambient_occlusion = tex2D( DecalOrAmbientOcclusionMapSampler, IN.Texcoord.zw ).a;
#endif

	// ........................................................................
	// shadow map
	// ........................................................................

	half shadow_test = 1;

#if SHADOW_QUALITY >= 0 && defined( ENABLE_SHADOWS )
#if SHADOW_QUALITY >= 2

	{
		// BICUBIC
		float2 uv = IN.TexcoordProj.xy / IN.TexcoordProj.w;
	
	#ifdef _X36
		// workaround for possible shader compiler bug
		float z = max( 0, -IN.TexcoordProj.z );
	#else
		float z = -IN.TexcoordProj.z;
	#endif

		float4 D[4] = 
		{
			float4( 
				Fetch2DWithOffset( ShadowMapSampler, uv, -1.5, -1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, -0.5, -1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +0.5, -1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +1.5, -1.5 ).x ),
			float4( 
				Fetch2DWithOffset( ShadowMapSampler, uv, -1.5, -0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, -0.5, -0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +0.5, -0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +1.5, -0.5 ).x ),
			float4( 
				Fetch2DWithOffset( ShadowMapSampler, uv, -1.5, +0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, -0.5, +0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +0.5, +0.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +1.5, +0.5 ).x ),
			float4( 
				Fetch2DWithOffset( ShadowMapSampler, uv, -1.5, +1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, -0.5, +1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +0.5, +1.5 ).x,
				Fetch2DWithOffset( ShadowMapSampler, uv, +1.5, +1.5 ).x )
		};
		
		half4 Q = half4( 1, 0, 1, 0 ) + half4( -1, +1, -1, +1 ) * frac( uv / CameraPos.w - .5 ).xxyy;
		Q = Q * Q;
		half4 WX = half4( 0, .5, .5, 0 ) + half4( +.25, -.25, -.25, +.25 ) * Q.xyxy;
		half4 WY = half4( 0, .5, .5, 0 ) + half4( +.25, -.25, -.25, +.25 ) * Q.zwzw;

		shadow_test = dot( half4( 
			dot( D[0] > z, WX ),
			dot( D[1] > z, WX ),
			dot( D[2] > z, WX ),
			dot( D[3] > z, WX ) ), WY );
	}

#else

	{
		// BILINEAR
		float2 uv = IN.TexcoordProj.xy / IN.TexcoordProj.w;
		float z = -IN.TexcoordProj.z;

		float4 D = float4( 
			Fetch2DWithOffset( ShadowMapSampler, uv, -.5, -.5 ).x,
			Fetch2DWithOffset( ShadowMapSampler, uv, +.5, -.5 ).x,
			Fetch2DWithOffset( ShadowMapSampler, uv, -.5, +.5 ).x,
			Fetch2DWithOffset( ShadowMapSampler, uv, +.5, +.5 ).x );

		half4 Q = half4( 1, 0, 1, 0 ) + half4( -1, +1, -1, +1 ) * frac( uv / CameraPos.w - .5 ).xxyy;
		half4 W = Q.zzww * Q.xyxy;

		shadow_test = dot( D > z, W );
	}

#endif // SHADOW_QUALITY >= 2
#endif // SHADOW_QUALITY >= 0 && defined( ENABLE_SHADOWS )

	// ........................................................................
	// sum up all lights components
	// ........................................................................

	// calculate primary light component
	half3 kDiffuse = 0;
	half3 kSpecular = 0;

	{
		half4 dataA = LightData[0];
		half4 dataB = LightData[1];
		half3 lightcolor = dataB.xyz * projector_color * shadow_test;

#if LIGHT_TYPE == 0 
// directional light
		half3 L = dataA.xyz;
		half attenuation = 1;
#elif LIGHT_TYPE == 1
// spot light
		half3 dx = dataA.xyz + IN.V;
		half d2 = dot( dx, dx );
		half3 L = normalize( dx );
		half attenuation = pow( saturate( 1 - d2 * dataA.w ), 2 ) / d2;
#endif

#ifdef ENABLE_DIRECTIONLESS_LIGHTING
		kDiffuse += lightcolor * attenuation;
#else
		half3 H = normalize( L + V );
		half dotNL = saturate( dot( N, L ) );
		float dotNH = saturate( dot( N, H ) );

		half D = .5h * ( specular_exponent + 1 ) * pow( dotNH, specular_exponent );
#if SHADER_QUALITY >= 1
		half dotLH = saturate( dot( L, H ) );
		half FG = .25h / ( pow( dotLH, 3 ) + FRESNEL_CAP );
#else
		half FG = .25h;
#endif
		half DFG = D * FG;
		half dao = saturate( lerp( 1, ambient_occlusion, FogData[1].w ) );
		kDiffuse = dao * lightcolor * attenuation * dotNL;
		kSpecular = dao * lightcolor * attenuation * dotNL * DFG * specular_color_gloss.xyz;
#endif
	}


	// color sum
 
	half opacity = 1;
	half linopacity = 1;

#ifdef ENABLE_WATER

    float3 screenpos = IN.TexcoordScreen.xyz / IN.TexcoordScreen.w;
	float2 screentexcoord = screenpos.xy * float2( .5, -.5 ) + float2( .5, .5 );

	float Zp = screenpos.z;
	float Zm = tex2D( DepthCaptureMapSampler, screentexcoord ).x;
	
	float Dp = WaterData[0].w / ( 1 - Zp );
	float Dm = WaterData[0].w / ( 1 - Zm );

	half3 dV = V - normalize( V + 0.01h * N );
	float2 dp = float2( dot( dV, View2World[0] ), 0 );

	float Zm2 = tex2D( DepthCaptureMapSampler, screentexcoord + dp ).x;	
	Dm = Zm2 > Zp ? WaterData[0].w / ( 1 - Zm2 ) : Dm;
	dp = Zm2 > Zp ? dp : 0;

	half T0 = saturate( exp2( Dp - Dm ) );	
	half T1 = 1 - pow( diffuse_color_alpha.a, 2 );	
	half Tfresnel = 1 - pow( invdotNV * specular_color_gloss.a, 4 );

	half3 I0 = WaterData[0] * kDiffuse;
	half3 I1 = diffuse_color_alpha * kDiffuse;

	half3 col;
	col = lerp( I0, 0, T0 );
	col = lerp( I1, col, T1 );
	col = kSpecular + col * Tfresnel;
		
#else
  
  #ifdef ENABLE_ALPHABLEND

    #ifdef ENABLE_ALPHA_AS_COVERAGE
	opacity = diffuse_color_alpha.a;
	#else
	opacity = pow( diffuse_color_alpha.a, 1 - pow( invdotNV, 2 ) );
	#endif

	linopacity = opacity * opacity;

  #endif

	half3 col = linopacity * diffuse_color_alpha.xyz * kDiffuse;
  
  #ifdef ENABLE_ALPHA_AS_COVERAGE
	col += linopacity * kSpecular;
  #else
	col += kSpecular;
  #endif

#endif

	// ........................................................................
	// Fog
	// ........................................................................

	col = col * IN.FogTransmittance;
	
	col *= FogData[2].w;

	// ........................................................................
	// Return final result
	// ........................................................................

	half4 result;
	result.rgb = saturate( _gamma( col ) );

#ifdef ENABLE_ALPHABLEND
	result.a = opacity;
#else
	result.a = diffuse_color_alpha.a;
#endif

#ifdef ENABLE_DEBUG_MODE

	float DebugMode = DebugData.x;
	float NumLights = DebugData.y;

	if( DebugMode > 0 )
	{
		switch( DebugMode )
		{
		case 18:
			result.rgb = _gamma( kDiffuse );
			break;

		case 19:
			result.rgb = _gamma( kSpecular );
			break;

		default:
			result.rgb = 0;
			break;
		}

		result.a = 1;
	}

#endif

	return result;
}



// ------------------------------------------------------------------------
// Primary Render effect
// ------------------------------------------------------------------------


technique t0
{
	pass p0
	{
        VertexShader = compile VS_PROFILE VS_Default(); 
        PixelShader = compile PS_PROFILE PS_Default();
	}
}


